async function getApiUrl() {
  const localhostUrl = "http://localhost:3000";
  const productionUrl = "https://www.factaurial.fr";

  async function tryUrl(url, timeout = 2000) {
    try {
      const controller = new AbortController();
      const timeoutId = setTimeout(() => controller.abort(), timeout);

      const response = await fetch(`${url}/api/config`, {
        signal: controller.signal,
        mode: "cors",
      });
      clearTimeout(timeoutId);

      if (response.ok) {
        const data = await response.json();
        console.log(
          `[Extension] Successfully connected to ${url}, appUrl:`,
          data.appUrl
        );
        return data.appUrl || url;
      } else {
        console.log(`[Extension] ${url} returned status:`, response.status);
      }
    } catch (error) {
      if (error.name === "AbortError") {
        console.log(`[Extension] Timeout connecting to ${url}`);
      } else {
        console.log(`[Extension] Error connecting to ${url}:`, error.message);
      }
      return null;
    }
    return null;
  }

  console.log("[Extension] Trying localhost first...");
  const localResult = await tryUrl(localhostUrl, 1000);
  if (localResult) {
    console.log("[Extension] Using localhost:", localResult);
    return localResult;
  }

  console.log(
    "[Extension] Localhost failed, trying production:",
    productionUrl
  );
  const prodResult = await tryUrl(productionUrl, 5000);
  if (prodResult) {
    console.log("[Extension] Using production:", prodResult);
    return prodResult;
  }

  console.error("[Extension] Both localhost and production failed");
  console.error("[Extension] Production URL attempted:", productionUrl);

  throw new Error(
    `Impossible de se connecter à ${localhostUrl} (local) ni à ${productionUrl} (production).\n\nVérifiez que :\n- L'URL de production est correcte dans extension/popup.js (ligne 3)\n- Le site Vercel est accessible\n- Les permissions CORS sont configurées`
  );
}

async function checkVPNConnection() {
  try {
    const response = await fetch(
      "https://app.sandbox.skello.io/users/sign_in",
      {
        method: "HEAD",
        credentials: "include",
      }
    );

    if (response.ok || response.status === 200) {
      return { valid: true };
    }

    return { valid: false };
  } catch (error) {
    return { valid: false };
  }
}

document.getElementById("extractBtn").addEventListener("click", async () => {
  const btn = document.getElementById("extractBtn");
  const statusDiv = document.getElementById("status");

  btn.disabled = true;
  statusDiv.className = "status info";
  statusDiv.textContent = "Extraction en cours...";

  let API_URL;
  try {
    API_URL = await getApiUrl();
    console.log("[Extension] Selected API URL:", API_URL);
  } catch (error) {
    statusDiv.className = "status error";
    statusDiv.textContent = `❌ ${error.message}`;
    btn.disabled = false;
    return;
  }

  // TODO: Réactiver la vérification VPN une fois l'endpoint API Skello disponible
  // const vpnCheck = await checkVPNConnection();
  // if (!vpnCheck.valid) {
  //   statusDiv.className = "status error";
  //   statusDiv.textContent = "❌ Accès refusé";
  //   btn.disabled = false;
  //   return;
  // }

  try {
    const domains = [
      "app.factorialhr.com",
      ".factorialhr.com",
      "api.factorialhr.com",
      ".api.factorialhr.com",
      "factorialhr.com",
    ];

    const allCookiesArrays = await Promise.all(
      domains.map((domain) => chrome.cookies.getAll({ domain }))
    );

    const allCookies = allCookiesArrays.flat();
    const uniqueCookies = Array.from(
      new Map(allCookies.map((c) => [c.name, c])).values()
    );

    const factorialCookies = uniqueCookies.filter((c) =>
      c.name.startsWith("_factorial_")
    );

    if (factorialCookies.length === 0) {
      const allCookieNames = uniqueCookies.map((c) => c.name).join(", ");
      const debugInfo = allCookieNames
        ? `Cookies trouvés: ${allCookieNames}. Recherche des cookies commençant par "_factorial_".`
        : "Aucun cookie trouvé sur app.factorialhr.com.";
      throw new Error(
        `Aucun cookie Factorial trouvé. ${debugInfo} Assurez-vous d'être connecté sur app.factorialhr.com dans un onglet actif.`
      );
    }

    const hasFactorialData = factorialCookies.some(
      (c) => c.name === "_factorial_data"
    );
    if (!hasFactorialData) {
      const cookieNames = factorialCookies.map((c) => c.name).join(", ");
      throw new Error(
        `Cookie _factorial_data manquant. Cookies Factorial trouvés: ${cookieNames}. Reconnectez-vous sur app.factorialhr.com.`
      );
    }

    const cookieString = factorialCookies
      .map((c) => `${c.name}=${c.value}`)
      .join("; ");

    console.log(
      "[Extension] Sending cookies to:",
      `${API_URL}/api/auth/cookies`
    );
    const response = await fetch(`${API_URL}/api/auth/cookies`, {
      method: "POST",
      headers: {
        "Content-Type": "application/json",
        "X-Extension-Request": "true",
      },
      body: JSON.stringify({
        cookies: cookieString,
        domain: "app.factorialhr.com",
        vpnVerified: true, // TODO: Mettre à false et réactiver la vérification VPN
      }),
    });

    if (!response.ok) {
      let errorMessage = "Erreur lors de l'envoi";
      try {
        const error = await response.json();
        errorMessage = error.error || errorMessage;
      } catch (e) {
        errorMessage = `Erreur HTTP ${response.status}: ${response.statusText}`;
      }
      throw new Error(errorMessage);
    }

    const { sessionId } = await response.json();

    statusDiv.className = "status success";
    statusDiv.textContent = "✅ Cookies envoyés avec succès !";

    setTimeout(() => {
      window.close();
      window.open(`${API_URL}/auth/callback?sessionId=${sessionId}`, "_blank");
    }, 1500);
  } catch (error) {
    statusDiv.className = "status error";
    let errorMessage = error.message;
    console.error("[Extension] Error:", error);
    if (error.message === "Failed to fetch" || error.name === "TypeError") {
      errorMessage = `Impossible de se connecter à ${API_URL || "le serveur"}. Vérifiez que :\n- L'URL de production est correcte dans extension/popup.js\n- Le serveur est accessible\n- Les permissions CORS sont configurées`;
    }
    statusDiv.textContent = `❌ ${errorMessage}`;
    btn.disabled = false;
  }
});
